#include "bitmap6.hpp"
#include "larrydib.h"

/* LARRYDIB.H just has these two prototypes of functions in LARRYDIB.C
extern HBITMAP ReadBitmapFile(char *filename);
extern int WriteBitmapFile( char *filename, HBITMAP hbm);
*/

//----------------------WindowBitmap------------

WindowBitmap::WindowBitmap(): //Need a do-nothing constructor.
_hwnd(NULL),
_hdc(NULL),
_hbitmap(NULL),
_hblankbrush(NULL)
{
}

WindowBitmap::WindowBitmap(HWND hwnd, COLORREF blankcol): //constructor
_hwnd(hwnd),
_blankcolor(blankcol)
{
	_CX = GetSystemMetrics(SM_CXSCREEN); //Actual size of screen
	_CY = GetSystemMetrics(SM_CYSCREEN); //measured in pixels.
	_allocate();
	Clear();
}

WindowBitmap::WindowBitmap(HWND hwnd, int cx, int cy, COLORREF blankcol):
 _hwnd(hwnd),
 _CX(cx),
 _CY(cy),
 _blankcolor(blankcol)
{
	_allocate();
	Clear();
}

WindowBitmap::WindowBitmap(const WindowBitmap &wbm)
{
	_copy(wbm);
}

WindowBitmap::WindowBitmap(const TransparentWindowBitmap &twbm)
{
	_copy(twbm);
}

WindowBitmap::~WindowBitmap() //destructor
{
	_free();
}

WindowBitmap& WindowBitmap::operator=(const WindowBitmap &wbm)
{
	_free();
	_copy(wbm);
	return *this;
}

WindowBitmap& WindowBitmap::operator=(const TransparentWindowBitmap &twbm)
{
	_free();
	_copy(twbm);
	return *this;
}

// WindowBitmap memory helper functions-----------------------

void WindowBitmap::_allocate()
{
	HDC hdc = GetDC(_hwnd);
	/* In general it seems that these CreateCompatible??? calls work
		the most reliably when they are called with an HDC argument which
		comes straight from  a screen HWND.  I've had problems in
		using a derived hdc as the argument --- "measuring from what I
		cut".  So we'll always "measure" from a screen standard. */
	_hdc = CreateCompatibleDC(hdc); //Starts with area of one pixel!
	_hbitmap = CreateCompatibleBitmap(hdc, _CX, _CY);
		//This is really a memory allocation call as bitmap has size.
	if (!_hbitmap) //If there's not enough memory, tell & bail.
	{
		MessageBox( _hwnd,
			(LPSTR)"Failure in Bitmap Allocation!", //Text in window
			(LPSTR)"Memory Problems!", //Caption
			MB_OK | MB_ICONEXCLAMATION ); //Button flags
		SendMessage( _hwnd, WM_DESTROY, 0, 0L ); //Goes to WndProc.
		return;
	}
	//Selecting _hbitmap here makes _hdc have _CX by _CY area.
	SelectObject(_hdc, _hbitmap);
	//Create the blankbrush.
	_hblankbrush = CreateSolidBrush(_blankcolor);
	//And get rid of the hdc you got in the first line.
	ReleaseDC(_hwnd, hdc);
}

void WindowBitmap::_copy(const WindowBitmap &wbm)
{ //Assume that it has already done a _free
	_hwnd = wbm._hwnd;
	_CX = wbm._CX;
	_CY = wbm._CY;
	_blankcolor = wbm._blankcolor;
	_allocate();
	BitBlt(_hdc, 0, 0, _CX, _CX, wbm._hdc, 0, 0, SRCCOPY);
}

void WindowBitmap::_copy(const TransparentWindowBitmap &twbm)
{
	int i,j;

	_hwnd = twbm._hwnd;
	_CX = twbm._CX;
	_CY = twbm._CY;
	_blankcolor = twbm._blankcolor;
	_allocate();
	BitBlt(_hdc, 0, 0, _CX, _CY, twbm._hdc, 0, 0, SRCCOPY);
	/* Now we undo what Fixup does.  We color the transparent pixels
		_blankcolor.  Which are those?  The ones that are WHITE in _hdc_mask
		and BLACK in _hdc.*/
	for (j=0; j<_CY; j++)
		for (i=0; i<_CX; i++)
		{
			if ( (GetPixel(twbm._hdc, i, j) == RGB(0,0,0)) &&
				(GetPixel(twbm._hdc_mask, i, j) == RGB(255,255,255)) )
				SetPixel(_hdc, i, j, _blankcolor);
		}
	// Finally, code in the _blankcolor color as the pixel at (0, 0).
	if (twbm._transparent_equals_corner_color)
		SetPixel(_hdc, 0, 0, _blankcolor);
}

void WindowBitmap::_free()
{
	/* First Delete the device context, and then i will be safe to delete the
	GDI objects that were selected into it.  Remember the rule is just
	that you can't delete a GDI tool while it is selected into a valid HDC. */
	if (_hdc)
		DeleteDC(_hdc);
	//Delete the bitmap.
	if (_hbitmap)
		DeleteObject(_hbitmap);
	//Delete the _hblankbrush.
	if (_hblankbrush)
		DeleteObject(_hblankbrush);
	_hdc = NULL;  //Set these to NULL so you can't delete twice
	_hbitmap = NULL;
	_hblankbrush = NULL;
}

//--------------WindowBitmap methods---------------

void WindowBitmap::Clear()
{ //This clears the bitmap.
	HBRUSH callerbrush;

	//Select the blankbrush.
	callerbrush = SelectObject(_hdc, _hblankbrush);
	//This covers the screen with the selected brush pattern.
	PatBlt(_hdc, 0, 0, _CX, _CY, PATCOPY);
	//Reselect the callerbrush, so you can delete _hblankbrush if need to.
	SelectObject(_hdc, callerbrush);
}

void WindowBitmap::SetBlankColor(COLORREF blankcol)
{
	DeleteObject(_hblankbrush);
	_hblankbrush = CreateSolidBrush(blankcol);
}

void WindowBitmap::CutFrom(HDC hdc, const Rect &rect)
{
	_free();
	_CX = rect.right() - rect.left();
	_CY = rect.bottom() - rect.top();
	_allocate();
	//Now copy the rect region of hdc to the (0,0) position of _hdc
	BitBlt(_hdc, 0, 0, _CX, _CY, hdc, rect.left(), rect.top(), SRCCOPY);
}

void WindowBitmap::LoadResourceBitmap(HINSTANCE hInst, char* resource_name)
{
	HBITMAP            new_hbitmap = NULL;    /* handle to the bitmap */
	BITMAP 				 bm;			/*Bitmap structure for getting size*/

	new_hbitmap = LoadBitmap( hInst, resource_name);
	if (!new_hbitmap) //If there's not enough memory, tell & bail.
	{
		MessageBox( _hwnd,
			(LPSTR)"Failure in Bitmap Loading!",
			(LPSTR)(LPSTR)resource_name,
			MB_OK | MB_ICONEXCLAMATION ); //Button flags
		return;
	}
	SelectObject(_hdc, new_hbitmap); //This unselects current _hbitmap
	DeleteObject(_hbitmap);
	_hbitmap = new_hbitmap;
	GetObject(_hbitmap, sizeof(BITMAP), (LPSTR)&bm);
	_CX = bm.bmWidth;
	_CY = bm.bmHeight;
}

void WindowBitmap::LoadResourceBitmap(HINSTANCE hInst, int resource_ID)
{
	LoadResourceBitmap( hInst, (char*)MAKEINTRESOURCE(resource_ID));
}

void WindowBitmap::LoadFileBitmap(LPSTR file_bitmap_name, BOOL resizeflag)
{
	WindowBitmap	*copy_wbm; //Use this in the NOT resizeflag case.
	HBITMAP new_hbitmap = NULL;    // handle to the bitmap
	BITMAP bm;			//Bitmap structure for getting size

	new_hbitmap = ReadBitmapFile((char *)file_bitmap_name);
		/* You can't allocate this guy as a WindowBitmap, as you have
		no idea what his size is! */
	if (!new_hbitmap) //If there's not enough memory, tell & bail.
	{
		MessageBox( _hwnd,
			(LPSTR)"Failure in Bitmap Loading!",
			(LPSTR)(LPSTR)file_bitmap_name,
			MB_OK | MB_ICONEXCLAMATION ); //Button flags
		return;
	}
	if (resizeflag)
	{
		SelectObject(_hdc, new_hbitmap); //This unselects current _hbitmap
		DeleteObject(_hbitmap);
		_hbitmap = new_hbitmap;
		GetObject(_hbitmap, sizeof(BITMAP), (LPSTR)&bm);
		_CX = bm.bmWidth;
		_CY = bm.bmHeight;
	}
	else   //don't resize the calling WindowBitmap.
	{
		copy_wbm = new WindowBitmap(_hwnd, 1, 1);
		SelectObject(copy_wbm->_hdc, new_hbitmap);
		DeleteObject(copy_wbm->_hbitmap);
		copy_wbm->_hbitmap = new_hbitmap;
		GetObject(copy_wbm->_hbitmap, sizeof(BITMAP), (LPSTR)&bm);
		copy_wbm->_CX = bm.bmWidth;
		copy_wbm->_CY = bm.bmHeight;
      copy_wbm->StretchTo(_hdc, Rect(0,0,_CX,_CY));
		delete copy_wbm;
	}
}

void WindowBitmap::SaveFileBitmap(LPSTR file_bitmap_name)
{
	WriteBitmapFile((char *)file_bitmap_name, _hbitmap);
}

void WindowBitmap::CopyTo(HDC hdc, const Rect &rect)
{
	BitBlt(hdc, rect.left(), rect.top(), //Next two arguments are width and height
		rect.right() - rect.left(), rect.bottom() - rect.top(),
		_hdc, rect.left(), rect.top(), SRCCOPY);
}

void WindowBitmap::StretchTo(HDC hdc, const Rect &rect)
{
	StretchBlt(hdc, rect.left(), rect.top(), //Next two arguments are width and height
		rect.right() - rect.left(), rect.bottom() - rect.top(),
		_hdc, 0, 0, _CX, _CY, SRCCOPY);
}

void WindowBitmap::CopyTo(HDC hdc, const RECT &rect)
{
	Rect myrect(rect);
	CopyTo(hdc, myrect);
}

void WindowBitmap::Repaint(HWND hwnd)
{
//It's faster to just write to the invalid part of the hwnd; this rectangle
//is stored in the PAINTSTRUCT of the hwnd as the rcPaint field.
	HDC hdc;
	PAINTSTRUCT ps;

	hdc = BeginPaint(hwnd, &ps);
	CopyTo(hdc, ps.rcPaint);
	EndPaint(hwnd, &ps);
}

void WindowBitmap::PasteTo(HDC hdc, const Point &point)
{ //Copy all of _hdc to the hdc rect with corner at point.
	BitBlt(hdc, point.x(), point.y(), _CX, _CY, _hdc, 0, 0, SRCCOPY);
}

//--------------TransparentWindowBitmap---------
TransparentWindowBitmap::TransparentWindowBitmap(): //Need a do-nothing constructor.
WindowBitmap(),
_hdc_mask(NULL),
_hbitmap_mask(NULL)
{
}

TransparentWindowBitmap::TransparentWindowBitmap(HWND hwnd, int cx,
	int cy, COLORREF transparent):
WindowBitmap(hwnd, cx, cy, transparent),
_transparent_equals_corner_color(TRUE)
{
	_allocate_child_fields();
}

TransparentWindowBitmap::TransparentWindowBitmap(const WindowBitmap &wbm):
WindowBitmap(wbm)
{
	_allocate_child_fields();
}

TransparentWindowBitmap::TransparentWindowBitmap(
	const TransparentWindowBitmap &twbm):
WindowBitmap() /*  We don't want to use the
	 WindowBitmap(TransparentWindowBitmap) constructor, so call the default */
{
	_allocate();
	_copy(twbm);
}

TransparentWindowBitmap::~TransparentWindowBitmap()
{
	_free();
}

TransparentWindowBitmap& TransparentWindowBitmap::operator=(const WindowBitmap &wbm)
{
	_free();
	_copy(wbm);
	return *this;
}

TransparentWindowBitmap& TransparentWindowBitmap::operator=(const TransparentWindowBitmap &twbm)
{
	_free();
	_copy(twbm);
	return *this;
}

//------TransparentWindowBitmap Helper functions -----

void TransparentWindowBitmap::_allocate()
{
	WindowBitmap::_allocate();
	_allocate_child_fields();
}

void TransparentWindowBitmap::_allocate_child_fields()
{
	HDC hdc = GetDC(_hwnd);

	_hdc_mask = CreateCompatibleDC(hdc); //Starts with area of one pixel!
	_hbitmap_mask = CreateCompatibleBitmap(hdc, _CX, _CY);
		//This is really a memory allocation call as bitmap has size.
	if (!_hbitmap_mask) //If there's not enough memory, tell & bail.
	{
		MessageBox( _hwnd,
			(LPSTR)"Failure in Bitmap Allocation!",
			(LPSTR)"Memory Problems!",
			MB_OK | MB_ICONEXCLAMATION ); //Button flags
		SendMessage( _hwnd, WM_DESTROY, 0, 0L ); //Goes to WndProc.
		return;
	}
	//Selecting _hbitmap here makes _hdc have _CX by _CY area.
	SelectObject(_hdc_mask, _hbitmap_mask);
	ReleaseDC(_hwnd, hdc);
}

void TransparentWindowBitmap::_copy(const WindowBitmap &wbm)
{  //Assume that it has already done a _free
	WindowBitmap::_copy(wbm);
	_allocate_child_fields();
	Fixup();
}

void TransparentWindowBitmap::_copy(const TransparentWindowBitmap &twbm)
{  //Assume that it has already done a _free
	_hwnd = twbm._hwnd;
	_CX = twbm._CX;
	_CY = twbm._CY;
	_blankcolor = twbm._blankcolor;
	_allocate();
	BitBlt(_hdc, 0, 0, _CX, _CX, twbm._hdc, 0, 0, SRCCOPY);
	BitBlt(_hdc_mask, 0, 0, _CX, _CX, twbm._hdc_mask, 0, 0, SRCCOPY);
}

void TransparentWindowBitmap::_free()
{
	WindowBitmap::_free();
	_free_child_fields();
}

void TransparentWindowBitmap::_free_child_fields()
{
	if (_hdc_mask)
		DeleteDC(_hdc_mask);
	if (_hbitmap_mask)
		DeleteObject(_hbitmap_mask);
	_hdc_mask = NULL;   //Set to NULL in case you do two frees.
	_hbitmap_mask = NULL;
}

//------TransparentWindowBitmap Methods-----------------
void TransparentWindowBitmap::SetBlankColor(COLORREF transparent)
{
	_blankcolor = transparent;
	WindowBitmap::SetBlankColor(transparent);
}

void TransparentWindowBitmap::Fixup()
{
	int i,j;
	COLORREF imagecolor;
	/*   To understand what we are doing
		here, and why, first look at TransparentWindowBitmap::PasteTo.
		Briefly, we want to be able to AND _hdc_mask with the target and then
		OR _hdc with the target, and have the image come through, with the
		"transparent" pixels left unchanged.  Fixup uses the current _hdc to
		make a fixup version of _hdc and a fixup _hdc_mask.  Wherever there
		is a "transparent" pixel in _hdc you want BLACK in the fixed _hdc,
		and you want WHITE in the _hdc_mask.  Wherever there is an image pixel
		(that is a non-transparent pixel) in _hdc, you want it the same in
		the fixup _hdc and you want it BLACK in _hdc_mask.*/
	for (j=0; j<_CY; j++)
		for (i=0; i<_CX; i++)
		{
			imagecolor = GetPixel(_hdc, i, j);
			if (imagecolor == _blankcolor) //Pixel is part of the background.
			{
				SetPixel(_hdc_mask, i, j, RGB(255,255,255));
				SetPixel(_hdc, i, j, RGB(0,0,0));
			}
			else //The pixel is part of the image.
			{
				SetPixel(_hdc_mask, i, j, RGB(0,0,0));
//				SetPixel(_hdc, i, j, imagecolor);  //This color is already in place!
			}
		}
}

void TransparentWindowBitmap::CutFrom(HDC hdc, const Rect &rect)
{ /* When you do a CutFrom here you should first do a
	SetBlankColor(backgroundcolor) for whatever the current screen background
	color is --- becuase it's possible you have the WRONG blankcolor in here
	as the result of a LoadFileBitmap call.  Of course if you have just
	constructed your TransparentWindowBimtmap with backgroundcolor as an
	argument then this isn't a problem. */
	_free_child_fields();
	WindowBitmap::CutFrom(hdc, rect);
	_allocate_child_fields();
	Fixup();
}

void TransparentWindowBitmap::LoadResourceBitmap(HINSTANCE hInst,
	LPSTR resource_bitmap_name)
{
	_free_child_fields();
	WindowBitmap::LoadResourceBitmap(hInst, resource_bitmap_name);
	_allocate_child_fields();
	if (_transparent_equals_corner_color)
			SetBlankColor(GetPixel(_hdc, 0, 0)); /* Asume blankcolor
		is equal to the corner color of the pixel image */
	Fixup();
}

void TransparentWindowBitmap::LoadResourceBitmap(HINSTANCE hInst,
	int resource_ID)
{
	LoadResourceBitmap( hInst, (char*)MAKEINTRESOURCE(resource_ID));
}

void TransparentWindowBitmap::LoadFileBitmap(LPSTR file_bitmap_name,
	 BOOL resizeflag)
{
	if (resizeflag)
		_free_child_fields();
	WindowBitmap::LoadFileBitmap(file_bitmap_name, resizeflag);
	if (resizeflag)
		_allocate_child_fields();
	if (_transparent_equals_corner_color)
			SetBlankColor(GetPixel(_hdc, 0, 0)); /* Asume blankcolor
		is equal to the corner color of the pixel image */
	Fixup();
}

void TransparentWindowBitmap::SaveFileBitmap(LPSTR file_bitmap_name)
{
	/* Use a copy constructor to make a copy of this which has
	_blankcolor in the transparent regions.  We store _blankcolor
	in position (0,0), provided _transparent_equals_corner_color is TRUE. */
	WindowBitmap wbm_save(*this); /* Calls the
		 WindowBitmap(TransparentWindowBitmap&)copy constructor. */
	wbm_save.SaveFileBitmap((char *)file_bitmap_name); /* Calls the
		WindowBitmap::SaveFileBitmap function */
	/* When you exit this call, wbm_save goes out of scope and
		calls its own destructor. */
}

void TransparentWindowBitmap::PasteTo(HDC hdc, const Point &point)
{  /* You (or a method) call TransparentWindowBitmap::Fixup before this call.
		Fixup has prepared _hdc_mask to be BLACK where the image goes, and
		WHITE where the transparent background goes.  The SRCAND BitBlt (which
		does a bitwise AND between source and target) sets the pixels in
		the target image location to BLACK and leaves the others
		alone.  That is, it nukes exactly those pixels where the image is
		to go.  Fixup has prepared _hdc to have the correct colors in the
		image region, and to have BLACK in the transparent background.  The
		SRCPAINT (BitBlt (which does a bitwise OR between source and target)
		puts the image pixels in the target location and leaves the others
		alone. */
	BitBlt(hdc, point.x(), point.y(), _CX, _CY, _hdc_mask, 0, 0, SRCAND);
	BitBlt(hdc, point.x(), point.y(), _CX, _CY, _hdc, 0, 0, SRCPAINT);
}

void TransparentWindowBitmap::CopyTo(HDC hdc, const Rect &rect)
{ // This uses the same idea as TransparentWindowBitmap::PasteTo.
	BitBlt(hdc, rect.left(), rect.top(),
		rect.right() - rect.left(), rect.bottom() - rect.top(),
		_hdc_mask, rect.left(), rect.top(), SRCAND);
	BitBlt(hdc, rect.left(), rect.top(),
		rect.right() - rect.left(), rect.bottom() - rect.top(),
		_hdc, rect.left(), rect.top(), SRCPAINT);
}


